// Functions only, no DOMContentLoaded event listener here.
// This file is meant for use in various page scripts.
// jQuery is required for this script.

class CUN2 {

    // utils
    constructor(
        thread_id = 0,
        thread_author_pk = null,
        my_pk = null,
        server_url = null,
        thread_container_id = "thread-container",
        reply_container_id = "reply-container",
        composer_container_id = "composer-container"
    ) {
        this.thread_id = thread_id;
        this.thread_author_pk = thread_author_pk;
        this.my_pk = my_pk;
        this.server_url = server_url || window.location.origin; // default to current origin (web app mode)
        this.thread_container_id = thread_container_id;
        this.reply_container_id = reply_container_id;
        this.composer_container_id = composer_container_id;
        this.submit_reply_function = null;
        this.highlighted_chat_id = null;
        this.auto_scroll = true;
        this.reply_scroll_position = 0;
        this.new_messages = 0;
        this.polling_interval = 2000; // 2 seconds
        this.chat_slide_duration = 100; // 0.3 seconds
        this.chat_slide_count = 5; // 5*100 = 0.5 seconds
    }

    start = function(submit_reply_function = null) {
        if (submit_reply_function && typeof submit_reply_function === "function") {
            this.submit_reply_function = submit_reply_function;
            this.build_composer_form();
        } // else, do nothing, likely in embed mode
        this.load_replies();

        // Listen to scroll up event on reply-container and set auto_scroll to false
        $(`#${this.reply_container_id}`).on('scroll', () => {
            const reply_container = $(`#${this.reply_container_id}`);
            this.reply_scroll_position = !isNaN(this.reply_scroll_position) ? this.reply_scroll_position : reply_container.scrollTop();
            const this_scroll_position = reply_container.scrollTop();
            const delta = this.reply_scroll_position - this_scroll_position;
            if (delta > 0) { // User scrolled up
                this.auto_scroll = false;
            } else { // User scrolled down
                // determine if user has scrolled to bottom
                const padding_account = 30;
                if (this_scroll_position + reply_container.height() + padding_account >= reply_container[0].scrollHeight) { // User scrolled to bottom
                    this.scroll_to_bottom(); // just clears new messages and sets auto_scroll to true in this case (scroll is not needed)
                }
            }
            this.reply_scroll_position = this_scroll_position * 1;
        });

        // initialize the live lamports updater
        setTimeout(init_live_lamports, 50);
    }

    end_session = function() {
        this.my_pk = null;
        this.build_composer_form();
    }

    bs_icon = function(name = null) { // FOSS icons from https://icons.getbootstrap.com/
        const icons = {
            "list": // https://icons.getbootstrap.com/icons/list/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-list" viewBox="0 0 16 16">
                  <path fill-rule="evenodd" d="M2.5 12a.5.5 0 0 1 .5-.5h10a.5.5 0 0 1 0 1H3a.5.5 0 0 1-.5-.5m0-4a.5.5 0 0 1 .5-.5h10a.5.5 0 0 1 0 1H3a.5.5 0 0 1-.5-.5m0-4a.5.5 0 0 1 .5-.5h10a.5.5 0 0 1 0 1H3a.5.5 0 0 1-.5-.5"/>
                </svg>`,
            "bolt": // https://icons.getbootstrap.com/icons/lightning-charge-fill/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-lightning-charge-fill" viewBox="0 0 16 16">
                  <path d="M11.251.068a.5.5 0 0 1 .227.58L9.677 6.5H13a.5.5 0 0 1 .364.843l-8 8.5a.5.5 0 0 1-.842-.49L6.323 9.5H3a.5.5 0 0 1-.364-.843l8-8.5a.5.5 0 0 1 .615-.09z"/>
                </svg>`,
            "info": // https://icons.getbootstrap.com/icons/exclamation-diamond-fill/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-exclamation-diamond-fill" viewBox="0 0 16 16">
                  <path d="M9.05.435c-.58-.58-1.52-.58-2.1 0L.436 6.95c-.58.58-.58 1.519 0 2.098l6.516 6.516c.58.58 1.519.58 2.098 0l6.516-6.516c.58-.58.58-1.519 0-2.098zM8 4c.535 0 .954.462.9.995l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 4.995A.905.905 0 0 1 8 4m.002 6a1 1 0 1 1 0 2 1 1 0 0 1 0-2"/>
                </svg>`,
            "sliders": // https://icons.getbootstrap.com/icons/sliders/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-sliders" viewBox="0 0 16 16">
                  <path fill-rule="evenodd" d="M11.5 2a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3M9.05 3a2.5 2.5 0 0 1 4.9 0H16v1h-2.05a2.5 2.5 0 0 1-4.9 0H0V3zM4.5 7a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3M2.05 8a2.5 2.5 0 0 1 4.9 0H16v1H6.95a2.5 2.5 0 0 1-4.9 0H0V8zm9.45 4a1.5 1.5 0 1 0 0 3 1.5 1.5 0 0 0 0-3m-2.45 1a2.5 2.5 0 0 1 4.9 0H16v1h-2.05a2.5 2.5 0 0 1-4.9 0H0v-1z"/>
                </svg>`,
            "send": // https://icons.getbootstrap.com/icons/send-fill/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-send-fill" viewBox="0 0 16 16">
                  <path d="M15.964.686a.5.5 0 0 0-.65-.65L.767 5.855H.766l-.452.18a.5.5 0 0 0-.082.887l.41.26.001.002 4.995 3.178 3.178 4.995.002.002.26.41a.5.5 0 0 0 .886-.083zm-1.833 1.89L6.637 10.07l-.215-.338a.5.5 0 0 0-.154-.154l-.338-.215 7.494-7.494 1.178-.471z"/>
                </svg>`,
            "x": // https://icons.getbootstrap.com/icons/x-lg/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-x-lg" viewBox="0 0 16 16">
                  <path d="M2.146 2.854a.5.5 0 1 1 .708-.708L8 7.293l5.146-5.147a.5.5 0 0 1 .708.708L8.707 8l5.147 5.146a.5.5 0 0 1-.708.708L8 8.707l-5.146 5.147a.5.5 0 0 1-.708-.708L7.293 8z"/>
                </svg>`,
            "right": // https://icons.getbootstrap.com/icons/arrow-right-short/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-right-short" viewBox="0 0 16 16">
                  <path fill-rule="evenodd" d="M4 8a.5.5 0 0 1 .5-.5h5.793L8.146 5.354a.5.5 0 1 1 .708-.708l3 3a.5.5 0 0 1 0 .708l-3 3a.5.5 0 0 1-.708-.708L10.293 8.5H4.5A.5.5 0 0 1 4 8"/>
                </svg>`,
            "reply": // https://icons.getbootstrap.com/icons/chevron-double-right/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-chevron-double-right" viewBox="0 0 16 16">
                  <path fill-rule="evenodd" d="M3.646 1.646a.5.5 0 0 1 .708 0l6 6a.5.5 0 0 1 0 .708l-6 6a.5.5 0 0 1-.708-.708L9.293 8 3.646 2.354a.5.5 0 0 1 0-.708"/>
                  <path fill-rule="evenodd" d="M7.646 1.646a.5.5 0 0 1 .708 0l6 6a.5.5 0 0 1 0 .708l-6 6a.5.5 0 0 1-.708-.708L13.293 8 7.646 2.354a.5.5 0 0 1 0-.708"/>
                </svg>`,
            "block": // https://icons.getbootstrap.com/icons/eye-slash-fill/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-eye-slash-fill" viewBox="0 0 16 16">
                  <path d="m10.79 12.912-1.614-1.615a3.5 3.5 0 0 1-4.474-4.474l-2.06-2.06C.938 6.278 0 8 0 8s3 5.5 8 5.5a7 7 0 0 0 2.79-.588M5.21 3.088A7 7 0 0 1 8 2.5c5 0 8 5.5 8 5.5s-.939 1.721-2.641 3.238l-2.062-2.062a3.5 3.5 0 0 0-4.474-4.474z"/>
                  <path d="M5.525 7.646a2.5 2.5 0 0 0 2.829 2.829zm4.95.708-2.829-2.83a2.5 2.5 0 0 1 2.829 2.829zm3.171 6-12-12 .708-.708 12 12z"/>
                </svg>`,
            "view_mode": // https://icons.getbootstrap.com/icons/eyeglasses/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-eyeglasses" viewBox="0 0 16 16">
                  <path d="M4 6a2 2 0 1 1 0 4 2 2 0 0 1 0-4m2.625.547a3 3 0 0 0-5.584.953H.5a.5.5 0 0 0 0 1h.541A3 3 0 0 0 7 8a1 1 0 0 1 2 0 3 3 0 0 0 5.959.5h.541a.5.5 0 0 0 0-1h-.541a3 3 0 0 0-5.584-.953A2 2 0 0 0 8 6c-.532 0-1.016.208-1.375.547M14 8a2 2 0 1 1-4 0 2 2 0 0 1 4 0"/>
                </svg>`,
            "back": // https://icons.getbootstrap.com/icons/chevron-double-left/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor"
                    class="bi bi-chevron-double-left" viewBox="0 0 16 16">
                    <path fill-rule="evenodd"
                        d="M8.354 1.646a.5.5 0 0 1 0 .708L2.707 8l5.647 5.646a.5.5 0 0 1-.708.708l-6-6a.5.5 0 0 1 0-.708l6-6a.5.5 0 0 1 .708 0" />
                    <path fill-rule="evenodd"
                        d="M12.354 1.646a.5.5 0 0 1 0 .708L6.707 8l5.647 5.646a.5.5 0 0 1-.708.708l-6-6a.5.5 0 0 1 0-.708l6-6a.5.5 0 0 1 .708 0" />
                </svg>`,
            "refresh": // https://icons.getbootstrap.com/icons/arrow-clockwise/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor"
                    class="bi bi-arrow-clockwise" viewBox="0 0 16 16">
                    <path fill-rule="evenodd"
                        d="M8 3a5 5 0 1 0 4.546 2.914.5.5 0 0 1 .908-.417A6 6 0 1 1 8 2z" />
                    <path
                        d="M8 4.466V.534a.25.25 0 0 1 .41-.192l2.36 1.966c.12.1.12.284 0 .384L8.41 4.658A.25.25 0 0 1 8 4.466" />
                </svg>`,
            "share": // https://icons.getbootstrap.com/icons/share/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-share-fill" viewBox="0 0 16 16">
                  <path d="M11 2.5a2.5 2.5 0 1 1 .603 1.628l-6.718 3.12a2.5 2.5 0 0 1 0 1.504l6.718 3.12a2.5 2.5 0 1 1-.488.876l-6.718-3.12a2.5 2.5 0 1 1 0-3.256l6.718-3.12A2.5 2.5 0 0 1 11 2.5"/>
                </svg>`,
            "embed": // https://icons.getbootstrap.com/icons/file-earmark-code/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor"
                    class="bi bi-file-earmark-code" viewBox="0 0 16 16">
                    <path
                        d="M14 4.5V14a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V2a2 2 0 0 1 2-2h5.5zm-3 0A1.5 1.5 0 0 1 9.5 3V1H4a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h8a1 1 0 0 0 1-1V4.5z" />
                    <path
                        d="M8.646 6.646a.5.5 0 0 1 .708 0l2 2a.5.5 0 0 1 0 .708l-2 2a.5.5 0 0 1-.708-.708L10.293 9 8.646 7.354a.5.5 0 0 1 0-.708m-1.292 0a.5.5 0 0 0-.708 0l-2 2a.5.5 0 0 0 0 .708l2 2a.5.5 0 0 0 .708-.708L5.707 9l1.647-1.646a.5.5 0 0 0 0-.708" />
                </svg>`,
            "compact_down": // https://icons.getbootstrap.com/icons/chevron-compact-down/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-chevron-compact-down" viewBox="0 0 16 16">
                  <path fill-rule="evenodd" d="M1.553 6.776a.5.5 0 0 1 .67-.223L8 9.44l5.776-2.888a.5.5 0 1 1 .448.894l-6 3a.5.5 0 0 1-.448 0l-6-3a.5.5 0 0 1-.223-.67"/>
                </svg>`,
            "person_plus": // https://icons.getbootstrap.com/icons/person-plus/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-person-plus" viewBox="0 0 16 16">
                  <path d="M6 8a3 3 0 1 0 0-6 3 3 0 0 0 0 6m2-3a2 2 0 1 1-4 0 2 2 0 0 1 4 0m4 8c0 1-1 1-1 1H1s-1 0-1-1 1-4 6-4 6 3 6 4m-1-.004c-.001-.246-.154-.986-.832-1.664C9.516 10.68 8.289 10 6 10s-3.516.68-4.168 1.332c-.678.678-.83 1.418-.832 1.664z"/>
                  <path fill-rule="evenodd" d="M13.5 5a.5.5 0 0 1 .5.5V7h1.5a.5.5 0 0 1 0 1H14v1.5a.5.5 0 0 1-1 0V8h-1.5a.5.5 0 0 1 0-1H13V5.5a.5.5 0 0 1 .5-.5"/>
                </svg>`,
            "person_check": // https://icons.getbootstrap.com/icons/person-check/
                `<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-person-check" viewBox="0 0 16 16">
                  <path d="M12.5 16a3.5 3.5 0 1 0 0-7 3.5 3.5 0 0 0 0 7m1.679-4.493-1.335 2.226a.75.75 0 0 1-1.174.144l-.774-.773a.5.5 0 0 1 .708-.708l.547.548 1.17-1.951a.5.5 0 1 1 .858.514M11 5a3 3 0 1 1-6 0 3 3 0 0 1 6 0M8 7a2 2 0 1 0 0-4 2 2 0 0 0 0 4"/>
                  <path d="M8.256 14a4.5 4.5 0 0 1-.229-1.004H3c.001-.246.154-.986.832-1.664C4.484 10.68 5.711 10 8 10q.39 0 .74.025c.226-.341.496-.65.804-.918Q8.844 9.002 8 9c-5 0-6 3-6 4s1 1 1 1z"/>
                </svg>`,

            // Currency icons
            "usd": '$',
            "eur": '€',
            "gbp": '£',
            "jpy": '¥',
            "cny": '¥',
            "rub": '₽',
            "krw": '₩',
            "sol": // https://cryptologos.cc/logos/solana-sol-logo.svg
                `<svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
                     viewBox="0 0 397.7 311.7" style="enable-background:new 0 0 397.7 311.7;" xml:space="preserve">
                <style type="text/css">
                    .st0{fill:url(#SVGID_1_);}
                    .st1{fill:url(#SVGID_2_);}
                    .st2{fill:url(#SVGID_3_);}
                </style>
                <linearGradient id="SVGID_1_" gradientUnits="userSpaceOnUse" x1="360.8791" y1="351.4553" x2="141.213" y2="-69.2936" gradientTransform="matrix(1 0 0 -1 0 314)">
                    <stop  offset="0" style="stop-color:#00FFA3"/>
                    <stop  offset="1" style="stop-color:#DC1FFF"/>
                </linearGradient>
                <path class="st0" d="M64.6,237.9c2.4-2.4,5.7-3.8,9.2-3.8h317.4c5.8,0,8.7,7,4.6,11.1l-62.7,62.7c-2.4,2.4-5.7,3.8-9.2,3.8H6.5
                    c-5.8,0-8.7-7-4.6-11.1L64.6,237.9z"/>
                <linearGradient id="SVGID_2_" gradientUnits="userSpaceOnUse" x1="264.8291" y1="401.6014" x2="45.163" y2="-19.1475" gradientTransform="matrix(1 0 0 -1 0 314)">
                    <stop  offset="0" style="stop-color:#00FFA3"/>
                    <stop  offset="1" style="stop-color:#DC1FFF"/>
                </linearGradient>
                <path class="st1" d="M64.6,3.8C67.1,1.4,70.4,0,73.8,0h317.4c5.8,0,8.7,7,4.6,11.1l-62.7,62.7c-2.4,2.4-5.7,3.8-9.2,3.8H6.5
                    c-5.8,0-8.7-7-4.6-11.1L64.6,3.8z"/>
                <linearGradient id="SVGID_3_" gradientUnits="userSpaceOnUse" x1="312.5484" y1="376.688" x2="92.8822" y2="-44.061" gradientTransform="matrix(1 0 0 -1 0 314)">
                    <stop  offset="0" style="stop-color:#00FFA3"/>
                    <stop  offset="1" style="stop-color:#DC1FFF"/>
                </linearGradient>
                <path class="st2" d="M333.1,120.1c-2.4-2.4-5.7-3.8-9.2-3.8H6.5c-5.8,0-8.7,7-4.6,11.1l62.7,62.7c2.4,2.4,5.7,3.8,9.2,3.8h317.4
                    c5.8,0,8.7-7,4.6-11.1L333.1,120.1z"/>
                </svg>`
        }
        var icon = icons?.[name] || `<span style="color:red; small">[ICON_ERR]</span>`;
        if (icon.length < 10) { // likely text
            icon = `<span class="currency-icon">${icon}</span>`;
        }
        icon = $(icon);
        if (icon.is("svg")) {
            icon.attr("height", "1em");
        }
        icon.addClass("cun2-icon");
        return icon;
    }

    increment_new_messages = function() {
        if (this.auto_scroll) return; // user will be scrolled to bottom anyway
        this.new_messages += 1;
        const plur = this.new_messages == 1 ? '' : 's';
        const msg = `&nbsp;${this.new_messages} new message${plur}&nbsp;`;
        $('#new_message_dismiss').empty().append(this.bs_icon("compact_down"), msg, this.bs_icon("compact_down"));
        $('#new_message_dismiss').fadeIn(300);
    }

    clear_new_messages = function() {
        this.new_messages = 0;
        $('#new_message_dismiss').fadeOut(300);
    }

    scroll_to_bottom = function() {
        this.auto_scroll = true;
        this.clear_new_messages();
        const reply_container = $(`#${this.reply_container_id}`);
        $(reply_container).scrollTop(reply_container[0].scrollHeight);
    }

    copy_share_link = function(embed = false) {
        // copy current page url adding ?embed=1 or &embed=1
        const url = new URL(window.location);
        if (embed) url.searchParams.set('embed', '1');
        navigator.clipboard.writeText(url.toString());
        const verb = embed ? 'Embed' : 'Share';
        alert(`${verb} link copied to clipboard!`);
    }

    // async fetchers
    fetch_replies = async function(last_chat_id = null) {

        var endpoint = `${this.server_url}/api/replies/${this.thread_id}`;
        if (last_chat_id) endpoint += `?last_chat_id=${last_chat_id}`;
        // GET replies from free endpoint
        // Wait for response and return JSON, not a promise
        const replies = await fetch(endpoint, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json'
            }
        })
            .then(response => response.json())
            .then(data => {
                if (!data.ok) throw new Error(data.error?.message || 'Unknown error');
                return data;
            })
            .catch(error => {
                console.error('Error fetching replies:', error);
                return {};
            })
        return replies;
    }

    highlight = function(id = 0) { // scroll to and highlight chat
        this.highlighted_chat_id = id;
        $(`.chat[data-chat-id="${id}"]`).addClass('highlighted');
    }

    remove_highlight = function() {
        this.highlighted_chat_id = null;
        $(`.chat`).removeClass('highlighted');
    }

    load_replies = async function() {
        try {

            // Check if .chat elements exist in the thread container
            // If they do, get the highest chat_id and use it as the last_chat_id
            // Otherwise, do not set last_chat_id
            const chats = $(`.chat`);
            var last_chat_id = 0;
            chats.each(function() {
                const chat_id = parseInt($(this).attr('data-chat-id'));
                if (chat_id > last_chat_id) {
                    last_chat_id = chat_id;
                }
            });

            // Fetch replies from the server
            const data = await this.fetch_replies(last_chat_id);
            const authors = data.authors;
            const channels = data.channels || {};
            const replies = data.replies.sort((a, b) => a.chat_id - b.chat_id);
            const container = $(`#${this.reply_container_id}`);

            for (var i = 0; i < replies.length; i++) {
                var tip_class = '', tip_pill = '';
                const reply = replies[i];
                const author = authors?.[reply.author_public_key] || { display_name: 'Unknown' };
                if ((reply?.tip_lamports || 0) > 0) {
                    tip_class = ' tip_chat text-warning alert-warning';
                    tip_pill = `<span class="badge rounded-pill text-bg-warning tip_pill live_lamports show_sol float-end" data-lamports="${reply.tip_lamports}"></span>`;
                }
                if ($(`.chat[data-chat-id="${reply.chat_id}"]`).length > 0) {
                    continue;
                }

                // Build channel badge if reply has a channel
                var channelBadge = '';
                if (reply.channel_id && channels[reply.channel_id]) {
                    const channel = channels[reply.channel_id];
                    channelBadge = `<a href="/profile/${channel.owner_public_key}?channel=${encodeURIComponent(channel.name)}" class="badge rounded-pill bg-primary" title="${channel.description || 'Channel: ' + channel.name}">${channel.name}</a> `;
                }

                const replyElement = $(`
                <div class="chat${tip_class}" data-chat-id="${reply.chat_id}" data-author-pk="${reply.author_public_key}">
                    <span class="text-muted small">
                        ${reply.chat_id} 
                        <a class="ghostly user-profile-link" href="/profile/${reply.author_public_key}" title="${reply.author_public_key}" data-public-key="${reply.author_public_key}">
                            ${author.display_name}
                        </a>
                    </span>
                    <span class="action-link-container float-end"></span>
                    <br>
                    ${channelBadge}${reply.content}
                    <br>
                    <span class="reaction-container" data-chat-id="${reply.chat_id}">
                        <button class="btn btn-sm btn-outline-secondary text-muted" style="opacity:0.5;">
                            ⮝ ...
                         </button>
                        <button class="btn btn-sm btn-outline-secondary text-muted" style="opacity:0.5;">
                            ⮟ ...
                         </button>
                    </span>${tip_pill}
                </div>`);

                const replyLink = $(`<a class="action-btn reply-btn" data-chat-id="${reply.chat_id}" data-author-pk="${reply.author_public_key}"></a>`);
                replyLink.empty().append(this.bs_icon("reply"));
                replyLink.on('click', (e) => {
                    e.preventDefault();
                    const target = $(e.target).closest('.reply-btn');
                    this.build_composer_form(target.attr('data-chat-id'), target.attr('data-author-pk'));
                });

                // Add replyLink to action-link-container
                replyElement.find('.action-link-container').append(replyLink);

                // Trigger replyLink if mobile users swipe right on chat
                replyElement.on('swipe', (e) => {
                    e.preventDefault();
                    $(this).find('.reply-btn').trigger('click');
                });

                // Add replyElement to container or to parent chat if it's a reply to another chat
                if (reply.reply_to && reply.reply_to != this.thread_id) {
                    const parentChat = $(`.chat[data-chat-id="${reply.reply_to}"]`);
                    if (parentChat.length > 0) {
                        parentChat.first().append(replyElement);
                    } else {
                        container.append(replyElement);
                    }
                } else {
                    container.append(replyElement);
                }
            }
            update_lamport_elements();

            // Move this to highlighting and scrolling interval
            if (this.highlighted_chat_id) {
                const hchat = $(`.chat[data-chat-id="${this.highlighted_chat_id}"]`);
                if (hchat.length > 0) {
                    hchat.addClass('highlighted');
                    hchat[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
                    this.auto_scroll = false; // override auto scroll to allow highlighted chat to stay in view.
                    this.highlighted_chat_id = null; // prevent future scroll-to
                }
            } else if (this.auto_scroll) {
                // smooth scroll to bottom of reply-container
                setTimeout(() => {
                    container.animate(
                        { scrollTop: container[0].scrollHeight },
                        300
                    );
                }, 50);
            } else if (replies.length > 0) {
                this.increment_new_messages();
            }

            // Add follow links (requires follow.js)
            setTimeout(clone_follow_links, 50);

            // Finally, setTimeout to poll for new replies
            setTimeout(() => {
                this.load_replies(); // automatically pulls top chat id
            }, this.polling_interval);

        } catch (error) {
            console.error('Error loading replies:', error);
        }
    }

    build_composer_form = function(reply_to = null, author_pk = null) { // boostrap 5.3 required
        const composer_container = $(`#${this.composer_container_id}`);
        if (!this.my_pk) {
            const server_home = this.server_url || window.location.origin;
            composer_container.empty().append(`Please <a href="${server_home}">log in</a> to post replies.`);
            return;
        }
        if (composer_container.length < 1) {
            return; // likely in embed mode
        }
        if (!this.submit_reply_function || typeof this.submit_reply_function != 'function') {
            composer_container.empty().append(
                `<div class="alert alert-danger">No submit_reply_function provided.</div>`
            );
            return;
        }
        try {
            reply_to = reply_to || this.thread_id;
            author_pk = author_pk || this.thread_author_pk;
            let resetFormBtn = null;
            const verb = reply_to == this.thread_id ? "Post to Thread" : `Reply to Chat #${reply_to}`;

            // init form
            const replyForm = $(`
                <form id="replyForm">
                    &nbsp;<a id="new_message_dismiss" class="ghostly float-end"></a>
                    <br>
                    <input type="hidden" id="reply_to" name="reply_to" value="${reply_to}">
                    <input type="hidden" id="author_public_key" name="author_public_key" value="${author_pk}">
                    <div class="input-group" id="reply_form_input_group">
                        <textarea class="form-control" id="reply_content" name="reply_content" rows="2" placeholder="${verb}..." required></textarea>
                        <input type="number" class="form-control" id="tip_amount" name="tip_amount" placeholder="Tip (optional)" step="0.001"
                            min="0">
                    </div>
                    <span class="text-muted small"><span id="charCount">0</span>/300</span>
                    <span class="float-end text-muted small"><a class="btn btn-dark btn-sm action-btn" id="currency-toggle"></a> Tip amount in SOL</span>
                </form>`);

            // build form
            const currency_toggle = replyForm.find('#currency-toggle');
            currency_toggle.empty().append(this.bs_icon("sliders"), "&nbsp;", this.bs_icon("sol"));
            currency_toggle.on('click', () => {
                $('#currencyModal').remove();
                // Create a modal
                const modal = $(`
                    <div class="modal fade" id="currencyModal" tabindex="-1" aria-labelledby="currencyModalLabel" aria-hidden="true">
                        <div class="modal-dialog">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <h5 class="modal-title" id="currencyModalLabel">Tipping Currency</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                    </div>
                                    <div class="modal-body">
                                        <div class="list-group" id="currencyList"></div>
                                        Tipping currency selector is not implemented yet.
                                        <br><br>
                                        <span style="font-style:italic;">Coming Soon!</span>
                                    </div>
                                    <div class="modal-footer">
                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                        <button type="button" class="btn btn-primary" data-bs-dismiss="modal">Save changes</button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                `);
                $('body').append(modal);
                // show modal
                const currencyModal = new bootstrap.Modal($('#currencyModal'));
                currencyModal.show();
            });

            // textarea input listener
            const textarea = replyForm.find("#reply_content");
            textarea.on("keydown", function(event) {
                // Submit when Enter is pressed unless Alt+Enter is pressed
                if (event.keyCode === 13) {
                    if (event.altKey) { // insert newline
                        $(this).val($(this).val() + "\n");
                        return true;
                    } else {
                        event.preventDefault();
                        sendButton.trigger("click");
                        return false;
                    }
                }
            });
            textarea.on("input", function() {
                const charCount = $(this).val().length;

                replyForm.find("#charCount").text(charCount);
                if (charCount > 300) {
                    $(this).addClass("invalid");
                    sendButton.prop("disabled", true);
                    $('#charCount').addClass("text-danger");
                } else {
                    $(this).removeClass("invalid");
                    sendButton.prop("disabled", false);
                    $('#charCount').removeClass("text-danger");
                }
            });

            // tip amount input listener
            const tipAmount = replyForm.find("#tip_amount");
            tipAmount.on("input", function() {
                const val = $(this).val();
                if (!isNaN(val * 1) && val > 0) {
                    $(this).addClass("valid");
                    $('#send_btn').addClass("btn-success");
                } else {
                    $(this).removeClass("valid");
                    $('#send_btn').removeClass("btn-success");
                }
            });

            // new message dismiss listener
            const dismiss = replyForm.find("#new_message_dismiss");
            dismiss.on("click", () => {
                this.scroll_to_bottom();
            });

            // render
            $(`#${this.composer_container_id}`).empty().append(
                resetFormBtn,
                replyForm
            );

            // buttons
            const sendButton = $('<button id="send_btn" type="submit" class="btn btn-primary form-control" title="Send Reply"></button>');
            sendButton.empty().append(this.bs_icon("send"));
            $('#reply_form_input_group').append(sendButton);
            if (reply_to != this.thread_id) {

                // link below resets to thread mode link if in reply to chat mode
                resetFormBtn = $('<button id="reset_form_btn" class="btn btn-danger form-control" title="Reset to thread mode"></button>');
                resetFormBtn.empty().append(`<span style="font-size:0.5em;opacity:0.6;">${reply_to}</span>`, '<br>', this.bs_icon("x"), '<br>', `<span style="font-size:0.5em;">&nbsp;</span>`);
                resetFormBtn.on("click", () => { this.build_composer_form(); });
                $('#reply_form_input_group').prepend(resetFormBtn);
            }

            // call class user defined function on submit
            replyForm.on('submit', this.submit_reply_function);

            // focus on textarea
            textarea.focus();
        } catch (e) {
            console.error(e);
        }
    }
}