/* App State */
class AppState {
	constructor() {
		this.version			= null;
		try{
			const manifest 	= chrome.runtime.getManifest();
			this.version 	= manifest?.version;
		}catch(e){
			console.error(e);
		}
		this.state				= { // persisted state
			settings:			{}, // Extension specific settings, User settings are managed in web app.
			bookmarks:			{},
			thread_sort_mode:	'date_desc', // default to newest sort mode
		};

		// settings
		this.settingsDefault 	= {
		    server_url:				"https://solthreadappserver-larrygiroux.replit.app/", // fallback server url
			server_api_key:			'',
			font_size:				'0.7em',
		};
        this.settingsSchema 	= {
            server_url:				'string',
			server_api_key:			'string',
			font_size:				'string',
        };
		this.settingsDescriptions = {
			server_url:				'The URL of the server to send chats to.',
			server_api_key:			'Your API key for interfacing with the server.',
			font_size:				'General font size of extension text. Some elements may be larger or smaller.',
		};
		for (let key in this.settingsDefault) this.settingsSchema[key] = typeof this.settingsDefault[key];
		this.settingsSchema.server_url = 'string';

		// other session state variables (not persisted)
		this.lockedThreadId		= null; // set to id if locked
		this.waitingURL			= null; // used to track real url if thread is locked
		this.currentURL 		= null; // The current URL being viewed by the user.
		this.loadState();

		// Uses class CUN2 from thread.js
		this.threadApp 			= null; // set externally after thread.js is loaded
	}

	feed(arg, err = false){
		if(err) console.trace(arg);	// for debugging
		$('#feed_error').toggle((err? true: false));
		$('#feed').empty().append((arg.toString() || "&nbsp;"));
	}

	lockThread(chat_id = null){
		// Will prevent re-render on user navigation in browser.
		this.lockedThreadId = chat_id;
	}

	unlockThread(){
		// Will allow re-render on user navigation in browser.
		this.lockedThreadId = null;
	}

	bookmarkThread(thread_id = null, url = null, content = null, author = null){
		// save light-weight version of thread to bookmarks.
		try{
			this.state.bookmarks[thread_id] = {
				url:		url,
				content:	content,
				author:		author, // Author name/alias str, not public key
			};
			this.saveState();
		}catch(e){
			console.error(e);
		}
	}

	unbookmarkThread(thread_id = null){
		try{
			delete this.state.bookmarks[thread_id];
			this.saveState();
		}catch(e){
			console.error(e);
		}
	}

	buildBookmarkList(){
		// Build bookmark list in #gui
		return;
	}

	// Load the state from chrome.storage.local
	loadState() {
		chrome.storage.local.get(['current_user_url', 'settings', 'bookmarks', 'thread_sort_mode'], (result) => {
			if (chrome.runtime.lastError) {
				console.error('Error loading state:', chrome.runtime.lastError);
				return;
			}
			this.state.current_user_url = result.current_user_url 	|| '';
			this.state.settings 		= result.settings 			|| {};
			this.state.bookmarks 		= result.bookmarks 			|| {};
			this.state.thread_sort_mode	= result.thread_sort_mode 	|| 'date_desc'; // default to newest sort mode
			// Ensure all default settings are present if missing
			for (let key in this.settingsDefault) {
				if (!(key in this.state.settings)) {
					this.state.settings[key] = this.settingsDefault[key];
				}
			}
		});
	}

	// Save the current state to chrome.storage.local
	saveState() {
		chrome.storage.local.set({
			settings:			this.state.settings 		|| {},
			bookmarks:			this.state.bookmarks		|| {},
			thread_sort_mode:	this.state.thread_sort_mode || 'date_desc',
		}, () => {
			if (chrome.runtime.lastError) {
				console.error('Error saving state:', chrome.runtime.lastError);
			}
		});
	}

	// Create a thread
	createThread() {
		return;
	}

	// send chat or create threda (reply_to is zero)
	sendChat() {
		return;
	}
	
	clearSearch(){
		$('#ext_search').val('').trigger('keyup');
	}

    setCurrentThreadID(threadId = null){
        this.currentThreadID = (threadId && !isNaN(threadId*1))? threadId*1: null;
    }

	getCurrentThreadID(){
		return this?.currentThreadID || null;
	}

	loadingMsg(msg = null){
		msg = msg || 'Loading';
		$('#gui').empty().append(`<div class="loading_message">${msg}<span class="loading_dots">.</span></div>`);
	}

	applyFontSizeSetting(){
		const font_size = this.getSetting('font_size');
		if(!font_size || typeof font_size != 'string' || font_size.length < 1) return;
		// validate that the font size ends with em and is a number from 0.5 to 1.5
		const font_size_num = parseFloat(font_size.replace('em',''));
		if(isNaN(font_size_num) || font_size_num < 0.5 || font_size_num > 1.5 || !font_size.endsWith('em')) return;
		$('body').css({fontSize: font_size});
	}

	// Update settings
	updateSettings(newSettings) {
		let validSettings = {};
		let invalidParams = [];
		for (let key in newSettings) {
			if (this.settingsSchema[key]){
				if(typeof newSettings[key] === this.settingsSchema[key]){
					validSettings[key] = newSettings[key];
				}else if(this.settingsSchema[key] === 'number' && !isNaN(newSettings[key]*1)){
					if (key in this.settingsLimits){
						const settingLimits = this.settingsLimits[key];
						if (!Array.isArray(settingLimits) || settingLimits.length !== 2) continue;
						validSettings[key] = newSettings[key] < settingLimits[0]? settingLimits[0]: newSettings[key];
						validSettings[key] = newSettings[key] > settingLimits[1]? settingLimits[1]: newSettings[key];
					}else{
						validSettings[key] = newSettings[key]*1;
					}
				}else if(this.settingsSchema[key] === 'boolean' && ["true","false"].indexOf(newSettings[key].toString().toLowerCase()) > -1){
					validSettings[key] = newSettings[key].toString().toLowerCase() === 'true'? true: false;
				}else{
					invalidParams.push(key);
				}
			}else{
				invalidParams.push(key);
			}
		}
	
		if(invalidParams.length > 0){
			const invStr = invalidParams.join(', ');
			this.feed(`Invalid setting or type for parameter(s): ${invStr}`,true);
		}else{
			// merge partial settings with existing settings
			this.state.settings = { ...this.state.settings, ...validSettings }; 
			this.saveState();
			this.feed("Settings updated.")
		}
	}

	getSetting(key){
		if('settings' in this.state && this.state.settings && typeof this.state.settings == 'object' && key in this.state.settings){
			return this.state.settings[key];
		}
		return this.settingsDefault[key] || null;
	}
	
	getShortURL(){
		const url_len 	= this.getSetting('url_preview_max_len');
		const url 		= this.state.current_user_url || '';
		var shortUrl 	= url.substring(0,url_len);
		return url.length > url_len? shortUrl + "...": url + "";
	}

	fetchNotifications(){
		return;
	}
	
	buildSettingsForm() {
        $('#nav-close').show(300);
		$('#gui').empty().append('<h2>Extension Settings</h2>');

		// Create cancel button
		const cancelIcon = this.heroicon('x-mark') || '❌';
		const cancelLink = $(`<a href="#" class="cancel pull-right" id="exit_settings">${cancelIcon} Close</a>`);
		cancelLink.on('click', (e) => {
			e.preventDefault();
			this.getThreads();
		});
		$('#gui').append(cancelLink,'<br><br>');


		// Get alpha sorted keys from this.state.settings
		const sortedKeys = Object.keys(this.settingsDefault).sort();

		const settingsForm = $(`<form></form>`);

        for (var i=0; i<sortedKeys.length; i++) {
			const key  	= sortedKeys[i];

            var input 		= null;
			const desc 		= this.settingsDescriptions?.[key] || null;
			var label 		= `<label for="${key}" title="${(desc? desc: '')}">${key.replace(/_/g, ' ').toUpperCase()}</label>`;
            if (key == 'font_size') { // font size dropdown
				input 			= $(`<select name="font_size" title="${(desc? desc: '')}"></select>`);
				const options 	= ['0.5em','0.6em','0.7em','0.8em','0.9em','1em','1.1em','1.2em','1.3em','1.4em','1.5em'];
				const dflt		= this.state.settings?.[key] || this.settingsDefault?.[key] || null;
				for(var j=0; j<options.length; j++){
					const opt 		= options[j];
					const selected 	= dflt == opt? ' selected': '';
					input.append(`<option value="${opt}"${selected}>${opt}</option>`);
				}
			} else if(key == 'blur_setting') {
				input 			= $(`<select name="blur_setting" title="${(desc? desc: '')}"></select>`);
				const options 	= ['show','blur','hide'];
				const dflt		= this.state.settings?.[key] || this.settingsDefault?.[key] || null;
				for(var j=0; j<options.length; j++){
					const opt 		= options[j];
					const selected 	= dflt == opt? ' selected': '';
					input.append(`<option value="${opt}"${selected}>${opt}</option>`);
				}
			} else if (typeof this.settingsDefault[key] === 'boolean') { // checkbox
				label			= null;
				const is_true 	= key in this.state.settings? this.state.settings[key]: this.settingsDefault[key];
				const checked 	= is_true? ' checked': '';
				input 			= `<input type="checkbox" title="${(desc? desc: '')}" name="${key}"${checked}>&nbsp;<label for="${key}" title="${(desc? desc: '')}">${key.replace(/_/g, ' ').toUpperCase()}</label><br>`;
            } else if(Array.isArray(this.settingsDefault[key])){ // checkboxes
				input 			= $(`<div style="font-size:0.8em;" class="checkbox_group" name="${key}"></div>`);
				var checkbox_options = [];
				switch(key){
					case 'show_conversions':
						checkbox_options = [ // update with server values later
							'USD',
							'EUR',
							'GBP',
						]
						break;
					default:;
				}
				for(var j=0; j<checkbox_options.length; j++){
					const opt 		= checkbox_options[j];
					const checked 	= this.state.settings?.[key]?.includes(opt)? ' checked': '';
					const checkbox 	= $(`<input type="checkbox" title="${(desc? desc: '')}" name="${key}" value="${opt}"${checked}>`);
					checkbox.prop('checked',this.state.settings?.[key]?.includes(opt) || false);
					input.append(checkbox,opt,'<br>');
				}
				input.append('<br>');
			} else {
                const typ 	= typeof this.settingsDefault[key] === 'number' ? 'number' : 'text';
				const val 	= this.state.settings?.[key] || this.settingsDefault[key];
				input 		= $(`<input type="${typ}" title="${(desc? desc: '')}" name="${key}" value="${val}">`);
            }
			if(!input) continue; // skip if no input
			if(label) settingsForm.append(label,'<br>');
            settingsForm.append(input,'<br>');
			
			if(key == 'server_url'){
				// Get all server_urls from invoices in this.state.invoices
				var server_urls = [];
				var v = this.state.settings?.[key] || this.settingsDefault[key];
				for (let name in this.state.invoices) {
					if(this.state.invoices[name].server_url && typeof this.state.invoices[name].server_url == 'string' && this.state.invoices[name].server_url.length > 0 && this.state.invoices[name].server_url != v){
						server_urls.push(this.state.invoices[name].server_url);
					}
				}
				// Make server_urls unique and alpha sorted
				server_urls = [...new Set(server_urls)].sort();
				// Add a button to set the input value to each of the available server_urls
				server_urls.forEach(server_url => {
					const urlset = $(`<a data-url="${server_url}">Set to ${server_url}</a>`);
					urlset.on('click', (e) => {
						e.preventDefault();
						const target = $(e.currentTarget);
						const server_url = target.data('url');
						$('input[name="server_url"]').val(server_url);
					});
					settingsForm.append(urlset,'<br>');
				});
				settingsForm.append('<br>');
			}
        }

		// append submit button
		settingsForm.append(`<br><br><input type="submit" value="Save Settings"><br><br>`);
		settingsForm.on('submit', (e) => {
			e.preventDefault();
			for(var key in this.settingsDefault){
				const input = settingsForm.find(`[name="${key}"]`);
				if(input.length < 1) continue;
				const val = input.val();
				if(Array.isArray(this.settingsDefault[key])){
					const checkedBoxes = input.find('input[type="checkbox"]:checked');
					const checkedValues = [];
					checkedBoxes.each((i, el) => {
						checkedValues.push($(el).val());
					});
					this.updateSettings({[key]: checkedValues});
				}else if(this.settingsSchema?.[key] == 'boolean'){
					this.updateSettings({[key]: input.is(':checked')});
				}else{
					this.updateSettings({[key]: val});
				}
			}
			this.saveState();
			this.updateConversionRates();
			this.applyFontSizeSetting();
			$('#exit_settings').trigger('click');
		});
		$('#gui').append(settingsForm);
    }

	renderNotification(){
		return;
	}

	// Move to notifications page.
	// buildNotificationsForm() {
	// 	return;
	// }
}
